﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;

//////////////////////////////////////////////////////////////////
//【自機のコントロールに使用するクラス】
// 自機をマウスの移動に合わせ移動させる処理やビーム砲の発射処理、相手ビームとの衝突処理を実行
//////////////////////////////////////////////////////////////////
public class FighterController : MonoBehaviour
{
    /////////////////////////////////
    // 【変数の設定】
    /////////////////////////////////

    //【Unity Editorから設定するPublic変数】
    public GameObject prefab_Beam;  // ビーム砲のプレハブ
    public ScoreManager class_ScoreManager;  // ScoreManagerの関数呼び出し用
    public AudioClip sound_BigExplosion;  // 自機爆発の効果音
    public AudioClip sound_Shot;  // ビーム砲の発射音
    public AudioSource audioSource;  // オーディオソース

    //【このプログラム内で使用するPrivate変数】
    private Vector3 startPos; // 前フレームのマウスの位置
    private Vector3 endPos;  // 現フレームのマウスの位置
    private Vector3 difPos;  // 前フレームと現フレームのマウスの移動量


    /////////////////////////////////
    // 【関数の設定】
    /////////////////////////////////

    //【ゲーム開始時に１回だけ動く関数】
    //（Unityの標準関数）
    void Start()
    {
        // ビームを繰り返し発生させるコルーチンを呼ぶ
        StartCoroutine("Create_Beam");
    }


    //【毎フレームに１度動く関数】
    //（Unityの標準関数）
    void Update()
    {
        //【自機の移動の概要】/////////////////////////////////
        // １．マウスボタンを押した位置をスタート位置として取得
        // ２．マウスボタンを押したまま移動させた場合、その位置をエンド位置として取得
        //`（エンド位置と言ってもまだマウスボタンは放していない）
        // ３．マウスのスタート位置からエンド位置の移動距離と方向を取得
        // ４．それを自機の位置に反映させる
        // ５．今のエンド位置を次のフレームのマウスのスタート位置として上記の２．に戻る
        //////////////////////////////////////////////////////////////////

        // マウスボタンを押した場合実行
        //（スマホの場合、タップした場合）
        if (Input.GetMouseButtonDown(0))
        {
            // マウスボタンを押した時のマウスの位置をスタート位置として取得
            //（スマホの場合、タップした位置を取得）
            startPos = Camera.main.ScreenToWorldPoint(Input.mousePosition);
        }

        // マウスボタンを押し続けている場合実行
        //（スマホの場合、スワイプしている場合実行）
        if (Input.GetMouseButton(0))
        {
            // マウスボタンを押し続けている場合、現在のマウスの位置を取得
            //（スマホの場合、スワイプしている現在の位置を取得）
            endPos = Camera.main.ScreenToWorldPoint(Input.mousePosition);
            // マウスボタンを押したスタート位置から押し続けている現在の位置までの差分（方向・距離）を計算
            difPos = endPos - startPos;

            // 自機の現在位置にマウスの移動方向・距離を加えた位置が画面の大きさの範囲内にいる場合
            if (2.4f > transform.position.x + difPos.x && transform.position.x + difPos.x > -2.4f &&
                4.5f > transform.position.y + difPos.y && transform.position.y + difPos.y > -4.5f)
            {
                // 自機をその位置に移動させる
                transform.Translate(difPos);
            }
            
            // 移動後の自機の位置を次のフレームのマウスの移動方向・距離の計算するスタート位置として使用
            startPos = Camera.main.ScreenToWorldPoint(Input.mousePosition);
        }
    }


    //【自機のコライダーが他のコライダーに衝突した場合呼ばれる関数】
    //（Unityの標準関数）
    private void OnTriggerEnter2D(Collider2D other)
    {
        // 衝突した相手のゲームオブジェクトのタグがBeam_Enemyの場合
        if (other.gameObject.tag == "Beam_Enemy")
        {
            // ScoreManagerのGameOver関数を呼び出す
            class_ScoreManager.GameOver();
            // 自機爆発の効果音を１回流す
            audioSource.PlayOneShot(sound_BigExplosion);
            // 自機のゲームオブジェクトを削除する
            Destroy(gameObject);
        }
    }


    //【ビームを発生させるコルーチン】
    IEnumerator Create_Beam()
    {
        // ビームのプレハブをシーン上に作成する
        Instantiate(prefab_Beam, transform.position, Quaternion.identity);
        //ビーム砲発射音を１回流す
        audioSource.PlayOneShot(sound_Shot);
        // プログラムを指定秒停止させる
        yield return new WaitForSeconds(0.5f);

        // 再度ビームを発生させるコルーチンを呼ぶ
        StartCoroutine("Create_Beam");
    }
}
