﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;

//////////////////////////////////////////////////////////////////
//【敵機のコントロールに使用するクラス】
// 敵機を移動させる処理やビーム砲の発射処理、相手ビームとの衝突処理を実行
//////////////////////////////////////////////////////////////////
public class EnemyController : MonoBehaviour
{
    /////////////////////////////////
    // 【変数の設定】
    /////////////////////////////////

    //【Unity Editorから設定するPublic変数】
    public int enemy_Type;  // このプログラムをつけた敵の種類（1か2）
    public GameObject prefab_Beam;  // このプログラムをつけた敵が使用するビーム
    public AudioClip sound_SmallExplosion;  // 敵爆発の効果音

    //【このプログラム内で使用するPrivate変数】
    private float speed_X; // 敵機のX方向（横方向）の移動スピード（数字が小さいほど真っ直ぐ移動）
    private GameObject gameManager; // Scene上のGameManagerゲームオブジェクト
    private AudioSource audioSource;  // オーディオソース


    /////////////////////////////////
    // 【関数の設定】
    /////////////////////////////////

    //【敵が出現時に１回だけ動く関数】
    //（Unityの標準関数）
    void Start()
    {
        // Scene上のGameManagerゲームオブジェクトを取得
        //（自機の場合Public変数で事前に関連付けをさせたが、敵機はもともとScene上に
        // ないので敵機が自動作成された際にScene上から取得する）
        gameManager = GameObject.Find("GameManager");
        // GameManagerに付いているAudioSourceコンポーネントを取得
        audioSource = gameManager.GetComponent<AudioSource>();

        // 敵機のX方向（横方向）の移動スピードをランダムに設定
        // 敵機の出現位置が画面中心から見て右側の場合
        if (transform.position.x >= 0)
            // 斜め左に向けて移動するようにスピードを設定
            speed_X = Random.Range(-1.5f, -0.1f);

        // 敵機の出現位置が画面中心から見て左側の場合
        else
            // 斜め右に向けて移動するようにスピードを設定
            speed_X = Random.Range(0.1f, 1.5f);

        // ビームを繰り返し発生させるコルーチンを呼ぶ
        StartCoroutine("Create_Beam");
    }


    //【毎フレームに１度動く関数】
    //（Unityの標準関数）
    void Update()
    {
        // 敵の種類enemy_Typeによって処理を変える
        switch (enemy_Type)
        {
            // 敵の種類enemy_Typeが１の場合
            case 1:
                // 毎フレーム、X方向（横方向）とY方向（縦方向）に指定の距離だけ移動する
                //（X方向とY方向に スピード×時間＝距離 だけ移動）
                transform.Translate(speed_X * Time.deltaTime, -2.0f * Time.deltaTime, 0);
                break;

            // 敵の種類enemy_Typeが２の場合
            case 2:
                // 毎フレーム、X方向（横方向）とY方向（縦方向）に指定の距離だけ移動する
                //（X方向とY方向に スピード×時間＝距離 だけ移動）
                transform.Translate(speed_X * Time.deltaTime, -1.0f * Time.deltaTime, 0);
                // 右斜め、左斜め方向と交互に移動するため、画面の端まで行ったら反対方向に移動を始める
                if (transform.position.x < -2)
                    speed_X = Mathf.Abs(speed_X);
                else if (transform.position.x > 2)
                    speed_X = -1.0f * Mathf.Abs(speed_X);
                break;
        }
    }


    //【敵がビームか境界に衝突した際に呼ばれる関数】
    //（Unityの標準関数）
    private void OnTriggerEnter2D(Collider2D other)
    {
        // 衝突した相手のゲームオブジェクトのタグがBeam_Fighterの場合
        if (other.gameObject.tag == "Beam_Fighter")
        {
            // 敵の種類enemy_Typeによって処理を変える
            switch (enemy_Type)
            {
                // 敵の種類enemy_Typeが１の場合
                case 1:
                    // 指定のスコアを追加する関数を呼び出す
                    gameManager.GetComponent<ScoreManager>().AddScore(1);
                    break;

                // 敵の種類enemy_Typeが２の場合
                case 2:
                    // 指定のスコアを追加する関数を呼び出す
                    gameManager.GetComponent<ScoreManager>().AddScore(2);
                    break;
            }

            // 敵機爆発の効果音を１回流す
            audioSource.PlayOneShot(sound_SmallExplosion);
            // 敵機のゲームオブジェクトを削除する
            Destroy(gameObject);
        }

        // 衝突した相手のゲームオブジェクトのタグがBoaderの場合
        if (other.gameObject.tag == "Border")
        {
            // 敵機のゲームオブジェクトを削除する
            Destroy(gameObject);
        }
    }


    //【ビームを発生させるコルーチン】
    IEnumerator Create_Beam()
    {
        // ビームのプレハブをシーン上に作成する
        Instantiate(prefab_Beam, transform.position, Quaternion.identity);
        // プログラムを指定秒停止させる
        yield return new WaitForSeconds(0.5f);

        // 再度ビームを発生させるコルーチンを呼ぶ
        StartCoroutine("Create_Beam");
    }
}
