﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;

//////////////////////////////////////////////////////////////////
//【ビームのコントロールに使用するクラス】
// ビームを移動させる処理や自機・敵機との衝突処理を実行
//////////////////////////////////////////////////////////////////
public class BeamController : MonoBehaviour
{
    /////////////////////////////////
    // 【変数の設定】
    /////////////////////////////////

    //【定数の設定】（レベル設計で変更可能）
    const float SPEED_Y = 3.0f; // ビームのY方向（縦方向）のスピード

    //【Unity Editorから設定するPublic変数】
    public int beam_Type; // ビームのタイプは3種類（1, 2, 3）

    //【このプログラム内で使用するPrivate変数】
    private float speed_X; // ビームのX方向（横方向）のスピード
    private ScoreManager class_ScoreManager; // ScoreManagerの関数呼び出し用
    private SpriteRenderer targetRenderer; // SpriteRendererコンポーネント取得用


    /////////////////////////////////
    // 【関数の設定】
    /////////////////////////////////

    //【ビームが出現時に１回だけ動く関数】
    //（Unityの標準関数）
    void Start()
    {
        // Scene上のGameManagerゲームオブジェクトに付けられたScoreManagerを取得
        //（自機の場合Public変数で事前に関連付けをさせたが、ビームはもともとScene上に
        // ないのでビームが自動作成された際にScene上から取得する）
        class_ScoreManager = GameObject.Find("GameManager").GetComponent<ScoreManager>();
        // ビームのゲームオブジェクトに付けられたSpriteRendererコンポーネントを取得
        targetRenderer = GetComponent<SpriteRenderer>();

        // 斜めに飛ぶビームのX方向（横方向）のスピードをランダムに設定
        speed_X = Random.Range(-1.0f, 1.0f);
    }


    //【毎フレームに１度動く関数】
    //（Unityの標準関数）
    void Update()
    {
        // ビームのゲームオブジェクトが画面外に出た場合かゲームクリアした場合
        if (!targetRenderer.isVisible || class_ScoreManager.Get_GameClear())
        {
            // ビームのゲームオブジェクトを削除する
            Destroy(gameObject);
        }

        // ビームのタイプbeam_Typeによって処理を変える
        switch (beam_Type)
        {
            // ビームのタイプbeam_Typeが１の場合
            case 1:
                // 毎フレーム、Y方向（縦方向）に指定の距離だけ移動する
                //（Y方向に スピード×時間＝距離 だけ移動）
                transform.Translate(0, SPEED_Y * Time.deltaTime, 0);
                break;

            // ビームのタイプbeam_Typeが２の場合
            case 2:
                // 毎フレーム、Y方向（縦方向）に指定の距離だけ移動する
                //（Y方向に スピード×時間＝距離 だけ移動）
                transform.Translate(0, -SPEED_Y * Time.deltaTime, 0);
                break;

            // ビームのタイプbeam_Typeが３の場合
            case 3:
                // 毎フレーム、X方向（横方向）とY方向（縦方向）に指定の距離だけ移動する
                //（X方向とY方向に スピード×時間＝距離 だけ移動）
                transform.Translate(speed_X * Time.deltaTime, -SPEED_Y * Time.deltaTime, 0);
                break;
        }
    }


    //【ビームが敵か自機に衝突した際に呼ばれる関数】
    //（Unityの標準関数）
    private void OnTriggerEnter2D(Collider2D other)
    {
        // 衝突した相手のゲームオブジェクトのタグがEnemyの場合
        if (other.gameObject.tag == "Enemy")
        {
            // ビームのタイプbeam_Typeが１の場合
            if (beam_Type == 1)
            {
                // ビームのゲームオブジェクトを削除する
                Destroy(gameObject);
            }
        }

        // 衝突した相手のゲームオブジェクトのタグがPlayerの場合
        if (other.gameObject.tag == "Player")
        {
            // ビームのタイプbeam_Typeが２か３の場合
            if (beam_Type == 2 || beam_Type == 3)
            {
                // ビームのゲームオブジェクトを削除する
                Destroy(gameObject);
            }
        }
    }
}
